<?php
/**
 * Módulo Admin: Gestión de Suscripciones Stripe
 * - Menú "Control Horario" + submenú "Suscripciones Stripe"
 * - Tabla con empresas, filtros y acciones
 * - Ajuste de clave secreta de Stripe (opción en BD)
 */
if (!defined('ABSPATH')) exit;

if (!class_exists('CH_Stripe_Admin_Subscriptions')) {

class CH_Stripe_Admin_Subscriptions {

    const OPT_SECRET_KEY = 'ch_stripe_secret_key';

    /** cache rápido de columnas por tabla */
private $table_columns_cache = [];

private function table_has_column($table, $column) {
    global $wpdb;
    $cache_key = $table . '::' . $column;
    if (isset($this->table_columns_cache[$cache_key])) return $this->table_columns_cache[$cache_key];
    $exists = (bool) $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = %s AND COLUMN_NAME = %s",
        $table, $column
    ));
    $this->table_columns_cache[$cache_key] = $exists;
    return $exists;
}

private function get_cached_pm_status($empresa_id) {
    $opt = get_option('ch_pm_' . intval($empresa_id));
    if (is_array($opt)) return $opt; // ['has'=>0|1,'ts'=>'Y-m-d H:i:s']
    return null;
}

private function set_cached_pm_status($empresa_id, $has) {
    update_option('ch_pm_' . intval($empresa_id), [
        'has' => $has ? 1 : 0,
        'ts'  => current_time('mysql'),
    ], false);
}


    public function __construct() {
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_post_ch_stripe_save_key', [$this, 'handle_save_key']);
        add_action('admin_post_ch_stripe_action',   [$this, 'handle_row_action']);
    }

  public function register_admin_menu() {
    // Menú raíz (apunta al Dashboard real)
    add_menu_page(
        'Control Horarios',
        'Control Horario',
        'manage_options',
        'ch-control-horario',
        'ch_render_ch_stripe_dashboard', // <- dashboard (función global del archivo "dashboard")
        'dashicons-clock',
        58
    );

    // Submenú: Dashboard (misma página que el raíz)
    add_submenu_page(
        'ch-control-horario',
        'Dashboard CH Stripe',
        'Dashboard',
        'manage_options',
        'ch-control-horario',
        'ch_render_ch_stripe_dashboard'
    );

    // Submenú: Suscripciones
    add_submenu_page(
        'ch-control-horario',
        'Suscripciones Stripe',
        'Suscripciones',
        'manage_options',
        'ch-suscripciones-stripe',
        [$this, 'render_admin_page']
    );

    // Submenú: Recordatorios PM (nuevo)
    add_submenu_page(
        'ch-control-horario',
        'Recordatorios Método de Pago',
        'Recordatorios PM',
        'manage_options',
        'ch-recordatorios-pm',
        [$this, 'render_pm_reminders_page']
    );
}


    public function render_dashboard_stub() {
        echo '<div class="wrap"><h1>Control Horario</h1><p>Panel principal del plugin.</p></div>';
    }

    public function handle_save_key() {
        if (!current_user_can('manage_options')) wp_die('No permitido');
        check_admin_referer('ch_stripe_save_key');

        $key = isset($_POST['stripe_secret_key']) ? sanitize_text_field($_POST['stripe_secret_key']) : '';
        update_option(self::OPT_SECRET_KEY, $key);
        wp_redirect(add_query_arg(['page'=>'ch-suscripciones-stripe', 'updated'=>'1'], admin_url('admin.php')));
        exit;
    }

    public function handle_row_action() {
        if (!current_user_can('manage_options')) wp_die('No permitido');
        check_admin_referer('ch_stripe_row_action');

        global $wpdb;
        $t = 'mnkt_empresas';

        $empresa_id = isset($_POST['empresa_id']) ? intval($_POST['empresa_id']) : 0;
        $do         = isset($_POST['do']) ? sanitize_key($_POST['do']) : '';

        if (!$empresa_id || !$do) {
            wp_redirect(add_query_arg(['page'=>'ch-suscripciones-stripe', 'err'=>'1'], admin_url('admin.php')));
            exit;
        }

        switch ($do) {
            case 'mark_converted':
                $wpdb->update($t, [
                    'trial_status'   => 'convertido',
                    'estado_empresa' => 'activado',
                ], ['id'=>$empresa_id]);
                break;

            case 'deactivate':
                $wpdb->update($t, [
                    'estado_empresa' => 'desactivado',
                    'deactivated_at' => current_time('mysql'),
                ], ['id'=>$empresa_id]);
                break;

            case 'activate':
                $wpdb->update($t, [
                    'estado_empresa' => 'activado',
                ], ['id'=>$empresa_id]);
                break;

            case 'refresh_from_stripe':
                $this->refresh_from_stripe($empresa_id);
                break;
        }

        wp_redirect(add_query_arg(['page'=>'ch-suscripciones-stripe', 'done'=>$do], admin_url('admin.php')));
        exit;
    }

   private function refresh_from_stripe($empresa_id) {
    global $wpdb;
    $t = 'mnkt_empresas';
    $row = $wpdb->get_row($wpdb->prepare(
        "SELECT id, stripe_subscription_id, stripe_customer_id FROM {$t} WHERE id=%d",
        $empresa_id
    ));
    if (!$row) return;

    $secret = get_option(self::OPT_SECRET_KEY, '');
    if (!$secret) return;

    // Cargar SDK (por si aún no está)
    $sdk = CH_STRIPE_WH_PLUGIN_DIR . 'lib/stripe-php/init.php';
    if (file_exists($sdk)) require_once $sdk;
    if (!class_exists('\Stripe\StripeClient')) return;

    $has_pm = false; // <- lo que queremos determinar

    try {
        $client = new \Stripe\StripeClient($secret);

        // 1) Si hay subscription id, refrescamos estado
        if (!empty($row->stripe_subscription_id)) {
            $sub = $client->subscriptions->retrieve($row->stripe_subscription_id, []);
            if ($sub && !empty($sub->status) && $sub->status === 'active') {
                $wpdb->update($t, [
                    'trial_status'           => 'convertido',
                    'estado_empresa'         => 'activado',
                    'stripe_subscription_id' => $sub->id,
                ], ['id'=>$empresa_id]);
            }
        }

        // 2) Comprobar método de pago guardado
        if (!empty($row->stripe_customer_id)) {
            // a) mira default_payment_method en invoice_settings
            $cus = $client->customers->retrieve($row->stripe_customer_id, []);
            if ($cus && !empty($cus->invoice_settings) && !empty($cus->invoice_settings->default_payment_method)) {
                $has_pm = true;
            }

            // b) si aún no, lista PMs tipo card (1 basta)
            if (!$has_pm) {
                $methods = $client->paymentMethods->all([
                    'customer' => $row->stripe_customer_id,
                    'type'     => 'card',
                    'limit'    => 1,
                ]);
                $has_pm = !empty($methods->data);
            }
        }

        // 3) Persistimos el estado: si existen columnas, a la tabla; si no, a options.
        $has_col = $this->table_has_column($t, 'has_payment_method');
        if ($has_col) {
            $set = [
                'has_payment_method'         => $has_pm ? 1 : 0,
                'payment_method_last_checked'=> current_time('mysql'),
            ];
            $wpdb->update($t, $set, ['id'=>$empresa_id]);
        } else {
            $this->set_cached_pm_status($empresa_id, $has_pm);
        }

    } catch (\Throwable $e) {
        // log “suave”: no rompas la UI
        if (method_exists($this, 'log')) {
            $this->log('Stripe refresh ERROR', ['e'=>$e->getMessage(), 'empresa_id'=>$empresa_id]);
        }
    }
}

public function render_admin_page() {
    if (!current_user_can('manage_options')) wp_die('No permitido');

    global $wpdb;
    $t_empresas = 'mnkt_empresas';
    $t_rel      = 'mnkt_empresas_usuarios';
    $t_users    = 'mnkt_users';

    // ¿La tabla tiene la columna has_payment_method?
    $has_pm_column = $this->table_has_column($t_empresas, 'has_payment_method');

    // Filtros
    $q    = isset($_GET['s']) ? trim(sanitize_text_field($_GET['s'])) : '';
    $page = max(1, isset($_GET['paged']) ? intval($_GET['paged']) : 1);
    $ppp  = 20;
    $off  = ($page - 1) * $ppp;

    // Query base con join para email gestor (si existe)
$where = "e.metodo_pago = 'Tarjeta'";
    $params = [];
    if ($q !== '') {
        $where .= " AND (e.user_cif LIKE %s OR e.razon_social LIKE %s OR u.user_email LIKE %s)";
        $like = '%' . $wpdb->esc_like($q) . '%';
        $params[] = $like; $params[] = $like; $params[] = $like;
    }

    // Añade columnas extras si existen en BD
    $extra_cols = '';
    if ($has_pm_column) {
        $extra_cols = ", e.has_payment_method, e.payment_method_last_checked";
    }

    $sql = "
        SELECT SQL_CALC_FOUND_ROWS
            e.id, e.user_cif, e.razon_social, e.estado_empresa, e.trial_status,
            e.trial_started_at, e.trial_expires_at, e.metodo_pago,
            e.stripe_customer_id, e.stripe_subscription_id,
            u.user_email AS gestor_email
            {$extra_cols}
        FROM {$t_empresas} e
        LEFT JOIN {$t_rel} r ON r.empresa_id = e.id AND r.rol_en_empresa='Gestor'
        LEFT JOIN {$t_users} u ON u.ID = r.user_id
        WHERE {$where}
        GROUP BY e.id
        ORDER BY e.id DESC
        LIMIT %d OFFSET %d
    ";

    $rows  = $wpdb->get_results($wpdb->prepare($sql, array_merge($params, [$ppp, $off])));
    $total = (int)$wpdb->get_var("SELECT FOUND_ROWS()");

    $secret = get_option(self::OPT_SECRET_KEY, '');

    ?>
    <div class="wrap">
        <h1>Suscripciones Stripe</h1>

        <?php if (isset($_GET['updated'])): ?>
            <div class="notice notice-success"><p>Configuración guardada.</p></div>
        <?php endif; ?>
        <?php if (isset($_GET['done'])): ?>
            <div class="notice notice-success"><p>Acción ejecutada: <code><?php echo esc_html($_GET['done']); ?></code></p></div>
        <?php endif; ?>

        <h2 class="title">Ajustes de Stripe</h2>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-bottom:20px;">
            <?php wp_nonce_field('ch_stripe_save_key'); ?>
            <input type="hidden" name="action" value="ch_stripe_save_key">
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label>Clave secreta Stripe</label></th>
                    <td>
                        <input type="password" name="stripe_secret_key" value="<?php echo esc_attr($secret); ?>" class="regular-text" placeholder="sk_live_... / sk_test_...">
                        <p class="description">Se guarda en la base de datos. Necesaria para “Refrescar desde Stripe”.</p>
                    </td>
                </tr>
            </table>
            <?php submit_button('Guardar clave'); ?>
        </form>

        <hr/>

        <form method="get" style="margin: 12px 0;">
            <input type="hidden" name="page" value="ch-suscripciones-stripe">
            <p class="search-box">
                <label class="screen-reader-text" for="empresa-search-input">Buscar:</label>
                <input type="search" id="empresa-search-input" name="s" value="<?php echo esc_attr($q); ?>">
                <input type="submit" id="search-submit" class="button" value="Buscar">
            </p>
        </form>

        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>CIF</th>
                    <th>Razón social</th>
                    <th>Gestor (email)</th>
                    <th>Estado</th>
                    <th>Trial</th>
                    <th>Stripe</th>
                    <th>Método guardado</th> <!-- NUEVA COLUMNA -->
                    <th>Acciones</th>
                </tr>
            </thead>
            <tbody>
            <?php if ($rows): foreach ($rows as $r): ?>
                <tr>
                    <td><?php echo intval($r->id); ?></td>
                    <td><code><?php echo esc_html($r->user_cif); ?></code></td>
                    <td><?php echo esc_html($r->razon_social ?: '—'); ?></td>
                    <td><?php echo esc_html($r->gestor_email ?: '—'); ?></td>
                    <td><?php echo esc_html($r->estado_empresa); ?></td>
                    <td>
                        <?php
                            echo 'Estado: ' . esc_html($r->trial_status ?: '—') . '<br/>';
                            if ($r->trial_started_at) {
                                echo 'Inicio: ' . esc_html( date_i18n('d/m/Y', strtotime($r->trial_started_at)) ) . '<br/>';
                            }
                            if ($r->trial_expires_at) {
                                echo 'Fin: ' . esc_html( date_i18n('d/m/Y', strtotime(get_date_from_gmt($r->trial_expires_at))) );
                            }
                        ?>
                    </td>
                    <td style="font-size:12px;">
                        <?php
                            echo 'Customer: ' . ($r->stripe_customer_id ? '<code>'.esc_html($r->stripe_customer_id).'</code>' : '—') . '<br/>';
                            echo 'Sub: ' . ($r->stripe_subscription_id ? '<code>'.esc_html($r->stripe_subscription_id).'</code>' : '—');
                        ?>
                    </td>
                    <td>
                        <?php
                            if ($has_pm_column && isset($r->has_payment_method)) {
                                echo $r->has_payment_method ? 'Sí' : 'No';
                                if (!empty($r->payment_method_last_checked)) {
                                    echo '<br><span style="color:#777;font-size:11px;">' .
                                        esc_html( date_i18n('d/m/y H:i', strtotime($r->payment_method_last_checked)) ) .
                                    '</span>';
                                }
                            } else {
                                // sin columna en BD: lee caché de options (tras pulsar “Refrescar”)
                                $cached = $this->get_cached_pm_status($r->id);
                                if ($cached) {
                                    echo $cached['has'] ? 'Sí' : 'No';
                                    echo '<br><span style="color:#777;font-size:11px;">' .
                                        esc_html( date_i18n('d/m/y H:i', strtotime($cached['ts'])) ) .
                                    '</span>';
                                } else {
                                    echo '—';
                                }
                            }
                        ?>
                    </td>
                    <td>
                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                            <?php wp_nonce_field('ch_stripe_row_action'); ?>
                            <input type="hidden" name="action" value="ch_stripe_action">
                            <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                            <input type="hidden" name="do" value="refresh_from_stripe">
                            <button class="button" <?php disabled(empty($secret)); ?>>Refrescar</button>
                        </form>

                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                            <?php wp_nonce_field('ch_stripe_row_action'); ?>
                            <input type="hidden" name="action" value="ch_stripe_action">
                            <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                            <input type="hidden" name="do" value="mark_converted">
                            <button class="button">Marcar convertido</button>
                        </form>

                        <?php if ($r->estado_empresa === 'activado'): ?>
                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                                <?php wp_nonce_field('ch_stripe_row_action'); ?>
                                <input type="hidden" name="action" value="ch_stripe_action">
                                <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                                <input type="hidden" name="do" value="deactivate">
                                <button class="button button-secondary">Desactivar</button>
                            </form>
                        <?php else: ?>
                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                                <?php wp_nonce_field('ch_stripe_row_action'); ?>
                                <input type="hidden" name="action" value="ch_stripe_action">
                                <input type="hidden" name="empresa_id" value="<?php echo intval($r->id); ?>">
                                <input type="hidden" name="do" value="activate">
                                <button class="button button-primary">Reactivar</button>
                            </form>
                        <?php endif; ?>
                    </td>
                </tr>
            <?php endforeach; else: ?>
                <tr><td colspan="9">No hay resultados.</td></tr>
            <?php endif; ?>
            </tbody>
        </table>

        <?php
            // Paginación simple
            $total_pages = max(1, ceil($total / $ppp));
            if ($total_pages > 1):
        ?>
            <div class="tablenav">
                <div class="tablenav-pages">
                    <?php
                        $base = add_query_arg(['page'=>'ch-suscripciones-stripe','s'=>$q, 'paged'=>'%#%'], admin_url('admin.php'));
                        echo paginate_links([
                            'base'      => $base,
                            'format'    => '',
                            'current'   => $page,
                            'total'     => $total_pages,
                            'prev_text' => '&laquo;',
                            'next_text' => '&raquo;',
                        ]);
                    ?>
                </div>
            </div>
        <?php endif; ?>
    </div>
    <?php
}

   
}

// Instanciar módulo admin
new CH_Stripe_Admin_Subscriptions();

} // end guard class_exists
